/**
 * PAINEL DE RECARGAS 1.0 - Server
 * Servidor Express + WebSocket para automação de Remix Lovable
 */

const express = require('express');
const http = require('http');
const WebSocket = require('ws');
const path = require('path');
const fs = require('fs');

// Serviço de automação
const remixAutomationService = require('./services/remixAutomationService');

// Configuração
let config = {
    port: 3008,
    headless: false
};

// Carregar configuração do arquivo
const configPath = path.join(__dirname, '..', 'config.json');
if (fs.existsSync(configPath)) {
    try {
        const fileConfig = JSON.parse(fs.readFileSync(configPath, 'utf8'));
        config = { ...config, ...fileConfig };
    } catch (error) {
        console.error('[Config] Erro ao carregar config.json:', error.message);
    }
}

// Express App
const app = express();
const server = http.createServer(app);

// WebSocket Server
const wss = new WebSocket.Server({ server });

// Prevenir crash do servidor em erros não tratados
process.on('uncaughtException', (err) => {
    console.error('[CRITICAL] Erro não tratado:', err);
});
process.on('unhandledRejection', (reason, promise) => {
    console.error('[CRITICAL] Rejeição não tratada:', reason);
});

// Middleware
app.use(express.json());
app.use(express.static(path.join(__dirname, '..', 'public')));

// Broadcast para todos os clientes WebSocket
function broadcast(type, data) {
    // Debug log para verificar se mensagens estão chegando
    if (type === 'remix-log') {
        // console.log('[DEBUG] Broadcasting log:', data.message);
    } else {
        console.log(`[Broadcast] Enviando '${type}' para ${wss.clients.size} clientes.`);
    }

    const message = JSON.stringify({ type, data });
    wss.clients.forEach(client => {
        if (client.readyState === WebSocket.OPEN) {
            client.send(message);
        }
    });
}

// Configurar broadcast no serviço de automação
remixAutomationService.setBroadcast(broadcast);

// WebSocket Connection Handler
wss.on('connection', (ws) => {
    console.log('[WS] Cliente conectado. Total:', wss.clients.size);

    ws.on('close', () => {
        console.log('[WS] Cliente desconectado');
    });

    ws.on('message', (message) => {
        try {
            const parsed = JSON.parse(message);
            if (parsed.type === 'debug-log') {
                console.log(`[Client Debug] ${parsed.message}`);
            }
        } catch (e) {
            // Ignore format errors
        }
    });

    ws.on('error', (error) => {
        console.error('[WS] Erro:', error.message);
    });
});

// ==================== API ROUTES - REMIX ====================

// Iniciar automação de remix
app.post('/api/remix/start', async (req, res) => {
    try {
        console.log('[API] /start chamado. Body:', JSON.stringify(req.body));
        let { projectUrl, baseName, remixRepeatCount, headless, email, password } = req.body;

        // Se payload estiver vazio (chamada via Telegram Bot Button)
        // ESTRATÉGIA NOVÁ: Enviar sinal para o Frontend (navegador) clicar no botão "Iniciar".
        // Isso evita que o servidor precise ter os dados salvos no config.json.
        // O servidor vira apenas um "pass-through".
        if (!email && !password && !projectUrl) {
            console.log('[API] Checkpoint: Payload vazio. Enviando sinal trigger-start-frontend para o navegador.');

            // Broadcast sinal para o frontend
            try {
                // Precisamos chamar a função broadcast que foi passada para o service? 
                // A função `broadcast` está definida neste arquivo server.js, podemos chamá-la direto?
                // Sim, está no escopo global do módulo ou acessível.
                // Mas wait, a função 'broadcast' foi definida lá em cima. 
                // Vamos movê-la ou garantir que podemos usá-la.
                // Ela está definida como `function broadcast(...)` no escopo do modulo. Perfeito.
                broadcast('trigger-start-frontend', { timestamp: Date.now() });

                return res.json({ success: true, message: 'Sinal de início enviado para o navegador.' });
            } catch (e) {
                console.error('[API] Erro ao enviar sinal para frontend:', e);
                return res.json({ success: false, error: 'Erro ao comunicar com navegador.' });
            }
        }

        // Validação obrigatória
        if (!email || !password) {
            const msg = '⚠️ <b>Atenção:</b> Para iniciar pelo Telegram, você precisa salvar Email e Senha no painel web (ícone 💾).';
            console.log(`[API] ${msg}`);

            // Tentar notificar via Telegram se possível
            if (config.telegramBotToken && config.telegramChatId) {
                const https = require('https');
                const url = `https://api.telegram.org/bot${config.telegramBotToken}/sendMessage?chat_id=${config.telegramChatId}&text=${encodeURIComponent(msg)}&parse_mode=HTML`;
                https.get(url).on('error', () => { });
            }

            return res.status(400).json({
                success: false,
                error: 'Credenciais não encontradas. Salve-as no Painel Web.'
            });
        }

        if (!projectUrl) {
            return res.status(400).json({
                success: false,
                error: 'URL do projeto é obrigatória (e não foi encontrada no config.json)'
            });
        }

        if (remixAutomationService.isRunning()) {
            return res.status(400).json({
                success: false,
                error: 'Automação já está em execução'
            });
        }

        // Iniciar em background
        remixAutomationService.start({
            projectUrl,
            baseName: baseName || 'STORE',
            remixRepeatCount: remixRepeatCount || 5,
            headless: headless || false,
            email,
            password,
            // Injetar credenciais do Telegram da config do servidor
            telegramBotToken: config.telegramBotToken,
            telegramChatId: config.telegramChatId,
            currentPort: PORT, // Para controle de proxy
            // Passar URL da instância para identificação
            instanceUrl: `http://localhost:${PORT}`
        });

        res.json({ success: true, message: 'Automação de remix iniciada via API/Telegram' });

    } catch (error) {
        console.error('[API] Erro ao iniciar remix:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});

// Parar automação de remix
app.post('/api/remix/stop', (req, res) => {
    try {
        remixAutomationService.stop();
        res.json({ success: true, message: 'Automação parada' });
    } catch (error) {
        console.error('[API] Erro ao parar remix:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});

// Status da automação
app.get('/api/remix/status', (req, res) => {
    res.json({
        running: remixAutomationService.isRunning()
    });
});

// ==================== HEALTH CHECK ====================

app.get('/api/health', (req, res) => {
    res.json({
        status: 'ok',
        timestamp: new Date().toISOString(),
        version: '1.0.0'
    });
});

// ==================== START SERVER ====================

const PORT = process.env.PORT || config.port || 3008;

server.listen(PORT, () => {
    console.log('');
    console.log('========================================');
    console.log('   PAINEL DE RECARGAS 1.0');
    console.log('   Lovable Remix Automation');
    console.log('========================================');
    console.log(`   🌐 http://localhost:${PORT}`);
    console.log('========================================');
    console.log('');
});

// Graceful shutdown
process.on('SIGINT', () => {
    console.log('\n[Server] Encerrando...');
    remixAutomationService.stop();
    server.close(() => {
        console.log('[Server] Encerrado com sucesso');
        process.exit(0);
    });
});

process.on('SIGTERM', () => {
    console.log('\n[Server] Encerrando...');
    remixAutomationService.stop();
    server.close(() => {
        console.log('[Server] Encerrado com sucesso');
        process.exit(0);
    });
});
